import tkinter as tk
from tkinter import scrolledtext, messagebox
import os
import subprocess
import urllib.request
import zipfile
import shutil

class MinecraftLauncher:
    def __init__(self, root):
        self.root = root
        self.root.title("Minecraft Beta 1.8.1 启动器")
        self.root.geometry("500x400")
        self.root.resizable(False, False)
        
        # 创建UI组件
        self.create_widgets()
        
        # 检查Java
        self.check_java()
    
    def create_widgets(self):
        # 用户名标签和输入框
        tk.Label(self.root, text="用户名:").place(x=20, y=30)
        self.username_entry = tk.Entry(self.root, width=30)
        self.username_entry.place(x=100, y=30)
        
        # 按钮
        self.launch_button = tk.Button(self.root, text="启动游戏", command=self.launch_minecraft, width=15)
        self.launch_button.place(x=100, y=80)
        
        self.download_java_button = tk.Button(self.root, text="下载Java", command=self.download_java, width=15)
        self.download_java_button.place(x=220, y=80)
        
        # 进度条
        self.progress_var = tk.DoubleVar()
        self.progress_bar = tk.Scale(self.root, variable=self.progress_var, orient="horizontal", length=400, from_=0, to=100, showvalue=False)
        self.progress_bar.place(x=50, y=120)
        self.progress_label = tk.Label(self.root, text="进度: 0%")
        self.progress_label.place(x=200, y=100)
        
        # 日志区标签
        tk.Label(self.root, text="日志:").place(x=20, y=160)
        
        # 日志区
        self.log_text = scrolledtext.ScrolledText(self.root, width=55, height=10, wrap=tk.WORD)
        self.log_text.place(x=20, y=190)
        self.log_text.config(state=tk.DISABLED)
    
    def add_log(self, message):
        """添加日志信息"""
        self.log_text.config(state=tk.NORMAL)
        self.log_text.insert(tk.END, message + "\n")
        self.log_text.see(tk.END)
        self.log_text.config(state=tk.DISABLED)
    
    def check_java(self):
        """检查Java是否存在"""
        self.add_log("正在检查Java运行库...")
        java_path = os.path.join(os.getcwd(), "jre-legacy", "bin", "javaw.exe")
        if os.path.exists(java_path):
            self.add_log("Java运行库已检测到！")
            return True
        else:
            self.add_log("未检测到Java运行库，请点击'下载Java'按钮进行下载。")
            return False
    
    def download_file(self, url, filename):
        """自动下载文件"""
        self.add_log(f"开始下载: {url}")
        try:
            # 显示下载进度
            def reporthook(count, block_size, total_size):
                if total_size > 0:
                    percent = int(count * block_size * 100 / total_size)
                    if percent % 10 == 0:  # 每10%更新一次
                        self.add_log(f"下载进度: {percent}%")
            
            # 下载文件
            urllib.request.urlretrieve(url, filename, reporthook)
            self.add_log(f"下载完成: {filename}")
            return True
        except Exception as e:
            self.add_log(f"错误: 下载失败: {str(e)}")
            return False
    
    def extract_zip(self, zip_path, extract_path):
        """自动解压zip文件"""
        self.add_log(f"开始解压: {zip_path}")
        try:
            # 确保解压目录存在
            if not os.path.exists(extract_path):
                os.makedirs(extract_path)
            
            # 使用zipfile库解压
            with zipfile.ZipFile(zip_path, 'r') as zip_ref:
                # 获取文件总数
                total_files = len(zip_ref.namelist())
                extracted_files = 0
                
                for file_info in zip_ref.infolist():
                    # 解压文件
                    zip_ref.extract(file_info, extract_path)
                    extracted_files += 1
                    
                    # 显示解压进度
                    if total_files > 0:
                        percent = int(extracted_files * 100 / total_files)
                        if percent % 20 == 0:  # 每20%更新一次
                            self.add_log(f"解压进度: {percent}%")
            
            self.add_log(f"解压完成: {zip_path}")
            return True
        except zipfile.BadZipFile:
            self.add_log(f"错误: {zip_path} 不是有效的zip文件")
            return False
        except Exception as e:
            self.add_log(f"错误: 解压失败: {str(e)}")
            return False
    
    def download_java(self):
        """下载Java运行库"""
        self.add_log("开始下载Java运行库...")
        
        # 下载jre-legacy.zip
        zip_filename = "jre-legacy.zip"
        if not self.download_file("https://beta18jdkdownlink.ak47s.com.cn/jre-legacy.zip", zip_filename):
            self.add_log("Java运行库下载失败，请检查网络连接后重试")
            return
        
        # 尝试解压
        zip_path = os.path.join(os.getcwd(), zip_filename)
        if not self.extract_zip(zip_path, os.getcwd()):
            self.add_log("Java运行库解压失败，尝试使用备用方法")
            
            # 尝试下载7-Zip
            self.add_log("尝试下载7-Zip解压工具...")
            seven_zip_filename = "7Z.zip"
            if self.download_file("https://beta18jdkdownlink.ak47s.com.cn/7Z.zip", seven_zip_filename):
                seven_zip_path = os.path.join(os.getcwd(), seven_zip_filename)
                seven_zip_extract_path = os.path.join(os.getcwd(), "7z")
                if self.extract_zip(seven_zip_path, seven_zip_extract_path):
                    self.add_log("7-Zip下载并解压成功")
                    # 这里可以添加使用7-Zip解压jre-legacy.zip的逻辑
                    self.add_log("请使用7-Zip手动解压jre-legacy.zip文件")
                else:
                    self.add_log("7-Zip解压失败，请手动解压")
            else:
                self.add_log("7-Zip下载失败，请手动解压jre-legacy.zip文件")
            return
        
        # 检查解压是否成功
        if self.check_java():
            self.add_log("Java运行库安装成功！")
            # 清理临时文件
            try:
                os.remove(zip_path)
                self.add_log("临时文件已清理")
            except Exception as e:
                self.add_log(f"清理临时文件失败: {str(e)}")
        else:
            self.add_log("Java运行库安装失败，请检查解压是否正确")
    
    def get_java_version(self, java_path):
        """获取Java版本信息"""
        try:
            result = subprocess.run([java_path, "-version"], capture_output=True, text=True)
            if result.returncode == 0:
                output = result.stderr  # Java -version 输出到stderr
                return output
            return "未知版本"
        except:
            return "无法获取版本"
    
    def launch_minecraft(self):
        """启动Minecraft"""
        username = self.username_entry.get().strip()
        if not username:
            messagebox.showerror("错误", "请输入用户名")
            return
        
        if not self.check_java():
            messagebox.showerror("错误", "未检测到Java运行库，请先下载Java")
            return
        
        self.add_log(f"正在启动Minecraft Beta 1.8.1，用户: {username}")
        
        # 构建启动命令
        jar_path = os.path.join(os.getcwd(), "b1.8.1", "b1.8.1.jar")
        natives_path = os.path.join(os.getcwd(), "b1.8.1", "b1.8.1-natives")
        
        # 检查游戏文件是否存在
        if not os.path.exists(jar_path):
            self.add_log(f"错误: 找不到Minecraft主文件: {jar_path}")
            messagebox.showerror("错误", f"找不到Minecraft主文件: {jar_path}")
            return
        
        if not os.path.exists(natives_path):
            self.add_log(f"错误: 找不到本地库文件目录: {natives_path}")
            messagebox.showerror("错误", f"找不到本地库文件目录: {natives_path}")
            return
        
        # 尝试不同的Java路径
        java_paths = []
        
        # 1. 优先使用自带的32位Java
        if os.path.exists(os.path.join(os.getcwd(), "jre-legacy", "bin", "javaw.exe")):
            java_paths.append(os.path.join(os.getcwd(), "jre-legacy", "bin", "javaw.exe"))
        if os.path.exists(os.path.join(os.getcwd(), "jre-legacy", "bin", "java.exe")):
            java_paths.append(os.path.join(os.getcwd(), "jre-legacy", "bin", "java.exe"))
        
        # 2. 尝试系统Java
        java_paths.extend(["javaw", "java"])
        
        # 尝试不同的启动参数组合
        param_combinations = [
            # 标准参数
            [
                f"-Djava.library.path={natives_path}",
                "-cp", jar_path,
                "net.minecraft.client.Minecraft",
                username
            ],
            # 兼容性参数
            [
                "-Xmx512M",
                "-Xms256M",
                "-Dsun.java2d.nodraw=true",
                f"-Djava.library.path={natives_path}",
                "-cp", jar_path,
                "net.minecraft.client.Minecraft",
                username
            ],
            # 32位兼容参数
            [
                "-Xmx512M",
                "-Xms256M",
                "-D32",
                "-Dsun.java2d.nodraw=true",
                f"-Djava.library.path={natives_path}",
                "-cp", jar_path,
                "net.minecraft.client.Minecraft",
                username
            ],
            # 无参数（最小化）
            [
                "-cp", jar_path,
                "net.minecraft.client.Minecraft",
                username
            ]
        ]
        
        # 尝试所有组合
        success = False
        for java_path in java_paths:
            self.add_log(f"尝试使用Java路径: {java_path}")
            
            # 获取Java版本信息
            version_info = self.get_java_version(java_path)
            self.add_log(f"Java版本信息: {version_info}")
            
            for i, params in enumerate(param_combinations):
                command = [java_path] + params
                self.add_log(f"尝试启动命令 #{i+1}: {' '.join(command)}")
                
                try:
                    # 启动进程
                    process = subprocess.Popen(
                        command,
                        cwd=os.getcwd(),
                        stdout=subprocess.PIPE,
                        stderr=subprocess.PIPE
                    )
                    
                    # 等待几秒钟，检查进程是否仍在运行
                    import time
                    time.sleep(3)
                    
                    if process.poll() is None:
                        # 进程仍在运行，启动成功
                        self.add_log("Minecraft启动成功！")
                        success = True
                        # 关闭管道
                        process.stdout.close()
                        process.stderr.close()
                        break
                    else:
                        # 进程已退出，读取错误信息
                        stdout, stderr = process.communicate()
                        error_msg = stderr.decode('gbk', errors='ignore') or stderr.decode('utf-8', errors='ignore')
                        self.add_log(f"启动失败，错误信息: {error_msg}")
                except Exception as e:
                    self.add_log(f"启动失败: {str(e)}")
            
            if success:
                break
        
        if not success:
            self.add_log("所有启动尝试都失败了")
            messagebox.showerror("错误", 
                "Minecraft启动失败！\n\n"
                "可能的原因:\n"
                "1. Java版本不兼容（建议使用Java 6或Java 7 32位）\n"
                "2. 游戏文件不完整\n"
                "3. 本地库文件与当前Java版本不兼容\n\n"
                "请确保:\n"
                "1. 已安装Java 8 32位版本\n"
                "2. b1.8.1文件夹包含完整的游戏文件\n"
                "3. b1.8.1-natives文件夹包含正确的本地库文件")
            self.add_log("建议:")
            self.add_log("1. 确保已通过'下载Java'按钮成功安装Java")
            self.add_log("2. 检查游戏文件完整性")
            self.add_log("3. 尝试以管理员权限运行启动器")
            self.add_log("4. 检查防火墙是否阻止游戏运行")

if __name__ == "__main__":
    root = tk.Tk()
    app = MinecraftLauncher(root)
    root.mainloop()
